from flask_openapi3 import APIBlueprint, Tag
from pydantic import BaseModel
from app.models import User, db
from app.schemas import (
    UserCreate, UserUpdate, UserResponse,
    LoginSchema, TokenResponse, PaginatedResponse
)
from app.schemas.user import UserCreateSchema
from app.decorators import token_required, generate_token
from sqlalchemy.exc import IntegrityError
from app.config import Config
from app.errors import api_response, success_response, error_response, ErrorDetail, APIResponse

user_tag = Tag(name="Users", description="Operações com usuários")
user_bp = APIBlueprint("users", __name__, url_prefix="/api/users", abp_tags=[user_tag])

@user_bp.post(
    "/register",
    responses={
        "201": APIResponse,
        "400": APIResponse
    },
)
@api_response
def register(body: UserCreateSchema):
    """Registra um novo usuário"""
    errors = []
    # Validação de senha
    if body.password != body.confirm_password:
        errors.append(ErrorDetail(field="confirm_password", message="As senhas não conferem"))
    # Validação de campos obrigatórios
    if not body.username:
        errors.append(ErrorDetail(field="username", message="Campo obrigatório"))
    if not body.email:
        errors.append(ErrorDetail(field="email", message="Campo obrigatório"))
    if not body.password:
        errors.append(ErrorDetail(field="password", message="Campo obrigatório"))
    if errors:
        return error_response(
            message="Erro de validação",
            errors=errors,
            code=400
        )
        
    try:
        user = User(
            username=body.username,
            email=body.email,
            first_name=body.first_name,
            last_name=body.last_name,
            phone=body.phone,
            birth_date=body.birth_date
        )
        user.set_password(body.password)
        db.session.add(user)
        db.session.commit()
        
        user_data = {
            "id": user.id,
            "username": user.username,
            "email": user.email,
            "first_name": user.first_name,
            "last_name": user.last_name,
            "phone": user.phone,
            "birth_date": user.birth_date
        }
        
        return success_response(
            message="Usuário criado com sucesso",
            data=user_data,
            code=201
        )
        
    except IntegrityError:
        db.session.rollback()
        return error_response(
            message="E-mail ou usuário já cadastrado",
            errors=[ErrorDetail(field="email", message="E-mail ou usuário já cadastrado")],
            code=400
        )

@user_bp.post(
    "/login",
    responses={
        "200": APIResponse,
        "401": APIResponse
    },
)
@api_response
def login(body: LoginSchema):
    """Login de usuário"""
    user = User.query.filter_by(username=body.username).first()
    
    if user and user.check_password(body.password):
        token = generate_token(user.id)
        return {"access_token": token}
    
    return error_response(
        message="Credenciais inválidas",
        errors=[ErrorDetail(field="credentials", message="Usuário ou senha inválidos")],
        code=401
    )

class QueryParams(BaseModel):
    page: int = 1
    per_page: int = Config.ITEMS_PER_PAGE

@user_bp.get(
    "/",
    responses={
        "200": APIResponse,
        "401": APIResponse
    },
    security=[{"bearerAuth": []}]
)
@token_required
@api_response
def list_users(query: QueryParams):
    """Lista todos os usuários (paginado)"""
    pagination = User.query.paginate(page=query.page, per_page=query.per_page)
    
    items = []
    for user in pagination.items:
        user_dict = {
            "id": user.id,
            "username": user.username,
            "email": user.email,
            "created_at": user.created_at_str,
            "updated_at": user.updated_at_str
        }
        items.append(user_dict)
    
    data = {
        "items": items,
        "total": pagination.total,
        "page": query.page,
        "per_page": query.per_page,
        "total_pages": pagination.pages
    }
    
    return data

class UserIdPath(BaseModel):
    user_id: int

@user_bp.get(
    "/<int:user_id>",
    responses={
        "200": APIResponse,
        "401": APIResponse,
        "404": APIResponse
    },
    security=[{"bearerAuth": []}]
)
@token_required
@api_response
def get_user(path: UserIdPath):
    """Obtém um usuário específico"""
    user = User.query.get_or_404(path.user_id)
    
    user_data = {
        "id": user.id,
        "username": user.username,
        "email": user.email,
        "first_name": user.first_name,
        "last_name": user.last_name,
        "phone": user.phone,
        "birth_date": user.birth_date,
        "created_at": user.created_at_str,
        "updated_at": user.updated_at_str
    }
    
    return user_data

@user_bp.patch(
    "/<int:user_id>",
    responses={
        "200": APIResponse,
        "400": APIResponse,
        "401": APIResponse,
        "403": APIResponse,
        "404": APIResponse
    },
    security=[{"bearerAuth": []}]
)
@token_required
@api_response
def update_user(path: UserIdPath, body: UserUpdate):
    """Atualiza um usuário"""
    user = User.query.get_or_404(path.user_id)
    
    if not hasattr(update_user, 'current_user') or update_user.current_user.id != path.user_id:
        return error_response(
            message="Não autorizado para atualizar este usuário",
            errors=[ErrorDetail(field="authorization", message="Usuário não tem permissão")],
            code=403
        )
    
    try:
        if body.username is not None:
            user.username = body.username
        if body.email is not None:
            user.email = body.email
        if body.password is not None:
            user.set_password(body.password)
        
        db.session.commit()
        
        return {
            "id": user.id,
            "username": user.username,
            "email": user.email,
            "first_name": user.first_name,
            "last_name": user.last_name,
            "phone": user.phone,
            "birth_date": user.birth_date,
            "created_at": user.created_at_str,
            "updated_at": user.updated_at_str
        }
        
    except IntegrityError:
        db.session.rollback()
        return error_response(
            message="Erro ao atualizar usuário",
            errors=[ErrorDetail(field="unique", message="Usuário ou email já existe")],
            code=400
        )

@user_bp.delete(
    "/<int:user_id>",
    responses={
        "200": APIResponse,
        "401": APIResponse,
        "403": APIResponse,
        "404": APIResponse
    },
    security=[{"bearerAuth": []}]
)
@token_required
@api_response
def delete_user(path: UserIdPath):
    """Deleta um usuário"""
    if not hasattr(delete_user, 'current_user') or delete_user.current_user.id != path.user_id:
        return error_response(
            message="Não autorizado para deletar este usuário",
            errors=[ErrorDetail(field="authorization", message="Usuário não tem permissão")],
            code=403
        )
    
    user = User.query.get_or_404(path.user_id)
    
    try:
        db.session.delete(user)
        db.session.commit()
        
        return None
        
    except IntegrityError:
        db.session.rollback()
        return error_response(
            message="Não é possível deletar este usuário",
            errors=[ErrorDetail(field="foreign_key", message="Existem registros vinculados a este usuário")],
            code=400
        )