from flask_openapi3 import APIBlueprint, Tag
from pydantic import BaseModel
from app.models import Expense, Income, Category, db
from app.decorators import token_required
from app.schemas import BaseModel, MessageResponse, ExpenseResponse, IncomeResponse
from sqlalchemy import extract, func
from datetime import datetime
from dateutil.relativedelta import relativedelta
from app.errors import api_response, success_response, error_response, ErrorDetail, SuccessResponse, ErrorResponse

report_tag = Tag(name="Reports", description="Relatórios financeiros")
report_bp = APIBlueprint("reports", __name__, url_prefix="/api/reports", abp_tags=[report_tag])

class MonthlySummaryQuery(BaseModel):
    month: int = datetime.now().month
    year: int = datetime.now().year

class CategoryTotal(BaseModel):
    category: str
    total: float

class MonthlySummaryResponse(BaseModel):
    month: int
    year: int
    expenses: list[CategoryTotal]
    incomes: list[CategoryTotal]
    total_expenses: float
    total_incomes: float
    balance: float

@report_bp.get(
    "/monthly-summary",
    responses={
        "200": SuccessResponse,
        "401": ErrorResponse,
    },
    security=[{"bearerAuth": []}]
)
@token_required
@api_response
def monthly_summary(query: MonthlySummaryQuery):
    """Obtém um resumo mensal de despesas e receitas"""
    # Total de despesas por categoria
    expenses = db.session.query(
        Category.name,
        func.sum(Expense.amount).label('total')
    ).join(Expense).filter(
        Expense.user_id == monthly_summary.current_user.id,
        extract('month', Expense.date) == query.month,
        extract('year', Expense.date) == query.year
    ).group_by(Category.name).all()

    # Total de receitas por categoria
    incomes = db.session.query(
        Category.name,
        func.sum(Income.amount).label('total')
    ).join(Income).filter(
        Income.user_id == monthly_summary.current_user.id,
        extract('month', Income.date) == query.month,
        extract('year', Income.date) == query.year
    ).group_by(Category.name).all()

    # Total geral
    total_expenses = sum(expense.total for expense in expenses)
    total_incomes = sum(income.total for income in incomes)
    balance = total_incomes - total_expenses

    data = {
        "month": query.month,
        "year": query.year,
        "expenses": [{"category": e.name, "total": float(e.total)} for e in expenses],
        "incomes": [{"category": i.name, "total": float(i.total)} for i in incomes],
        "total_expenses": float(total_expenses),
        "total_incomes": float(total_incomes),
        "balance": float(balance)
    }

    return success_response(
        message="Resumo mensal gerado com sucesso",
        data=data,
        code=200
    )

class YearlySummaryQuery(BaseModel):
    year: int = datetime.now().year

class MonthlyValues(BaseModel):
    expenses: float = 0
    incomes: float = 0
    balance: float = 0

class YearlySummaryResponse(BaseModel):
    year: int
    monthly_summary: dict[int, MonthlyValues]
    total_expenses: float
    total_incomes: float
    yearly_balance: float

@report_bp.get(
    "/yearly-summary",
    responses={
        "200": SuccessResponse,
        "401": ErrorResponse,
    },
    security=[{"bearerAuth": []}]
)
@token_required
@api_response
def yearly_summary(query: YearlySummaryQuery):
    """Obtém um resumo anual de despesas e receitas"""
    # Total mensal de despesas
    expenses = db.session.query(
        extract('month', Expense.date).label('month'),
        func.sum(Expense.amount).label('total')
    ).filter(
        Expense.user_id == yearly_summary.current_user.id,
        extract('year', Expense.date) == query.year
    ).group_by(extract('month', Expense.date)).all()

    # Total mensal de receitas
    incomes = db.session.query(
        extract('month', Income.date).label('month'),
        func.sum(Income.amount).label('total')
    ).filter(
        Income.user_id == yearly_summary.current_user.id,
        extract('year', Income.date) == query.year
    ).group_by(extract('month', Income.date)).all()

    # Organiza os resultados por mês
    monthly_summary = {}
    for month in range(1, 13):
        monthly_summary[month] = {
            "expenses": 0,
            "incomes": 0,
            "balance": 0
        }

    for expense in expenses:
        monthly_summary[int(expense.month)]["expenses"] = float(expense.total)

    for income in incomes:
        monthly_summary[int(income.month)]["incomes"] = float(income.total)

    # Calcula o balanço para cada mês
    for month in monthly_summary:
        monthly_summary[month]["balance"] = (
            monthly_summary[month]["incomes"] - monthly_summary[month]["expenses"]
        )

    # Totais anuais
    total_expenses = sum(float(expense.total) for expense in expenses)
    total_incomes = sum(float(income.total) for income in incomes)
    yearly_balance = total_incomes - total_expenses

    data = {
        "year": query.year,
        "monthly_summary": monthly_summary,
        "total_expenses": total_expenses,
        "total_incomes": total_incomes,
        "yearly_balance": yearly_balance
    }

    return success_response(
        message="Resumo anual gerado com sucesso",
        data=data,
        code=200
    )

@report_bp.get(
    "/recurring-expenses",
    responses={
        "200": SuccessResponse,
        "401": ErrorResponse,
    },
    security=[{"bearerAuth": []}]
)
@token_required
@api_response
def recurring_expenses():
    """Lista todas as despesas recorrentes ativas"""
    expenses = Expense.query.filter(
        Expense.user_id == recurring_expenses.current_user.id,
        Expense.is_recurring == True,
        (Expense.recurrence_end_date >= datetime.now().date()) | (Expense.recurrence_end_date == None)
    ).order_by(Expense.date).all()

    items = []
    for expense in expenses:
        items.append({
            "id": expense.id,
            "description": expense.description,
            "amount": expense.amount,
            "date": expense.date.isoformat(),
            "is_recurring": expense.is_recurring,
            "recurrence_end_date": expense.recurrence_end_date.isoformat() if expense.recurrence_end_date else None,
            "category_id": expense.category_id,
            "user_id": expense.user_id,
            "created_at": expense.created_at_str,
            "updated_at": expense.updated_at_str
        })

    return success_response(
        message="Despesas recorrentes listadas com sucesso",
        data={"items": items},
        code=200
    )

@report_bp.get(
    "/recurring-incomes",
    responses={
        "200": SuccessResponse,
        "401": ErrorResponse,
    },
    security=[{"bearerAuth": []}]
)
@token_required
@api_response
def recurring_incomes():
    """Lista todas as receitas recorrentes ativas"""
    incomes = Income.query.filter(
        Income.user_id == recurring_incomes.current_user.id,
        Income.is_recurring == True,
        (Income.recurrence_end_date >= datetime.now().date()) | (Income.recurrence_end_date == None)
    ).order_by(Income.date).all()

    items = []
    for income in incomes:
        items.append({
            "id": income.id,
            "description": income.description,
            "amount": income.amount,
            "date": income.date.isoformat(),
            "is_recurring": income.is_recurring,
            "recurrence_end_date": income.recurrence_end_date.isoformat() if income.recurrence_end_date else None,
            "category_id": income.category_id,
            "user_id": income.user_id,
            "created_at": income.created_at_str,
            "updated_at": income.updated_at_str
        })

    return success_response(
        message="Receitas recorrentes listadas com sucesso",
        data={"items": items},
        code=200
    )

class CategoryTrendsQuery(BaseModel):
    months: int = 6  # Últimos 6 meses por padrão

@report_bp.get(
    "/category-trends",
    responses={
        "200": SuccessResponse,
        "401": ErrorResponse,
    },
    security=[{"bearerAuth": []}]
)
@token_required
@api_response
def category_trends(query: CategoryTrendsQuery):
    """Analisa tendências de gastos por categoria nos últimos meses"""
    start_date = datetime.now().date() - relativedelta(months=query.months)

    # Tendências de despesas por categoria
    expense_trends = db.session.query(
        Category.name,
        extract('month', Expense.date).label('month'),
        extract('year', Expense.date).label('year'),
        func.sum(Expense.amount).label('total')
    ).join(Expense).filter(
        Expense.user_id == category_trends.current_user.id,
        Expense.date >= start_date
    ).group_by(
        Category.name,
        extract('month', Expense.date),
        extract('year', Expense.date)
    ).order_by(
        Category.name,
        extract('year', Expense.date),
        extract('month', Expense.date)
    ).all()

    # Organiza os resultados por categoria
    trends = {}
    end_date = datetime.now().date()
    
    for trend in expense_trends:
        if trend.name not in trends:
            trends[trend.name] = {
                "category": trend.name,
                "trend": [],
                "average": 0,
                "total": 0
            }
        
        monthly_trend = {
            "month": int(trend.month),
            "year": int(trend.year),
            "total": float(trend.total)
        }
        trends[trend.name]["trend"].append(monthly_trend)
        trends[trend.name]["total"] += monthly_trend["total"]
    
    # Calcula a média mensal para cada categoria
    for category in trends.values():
        category["average"] = category["total"] / len(category["trend"]) if category["trend"] else 0
    
    data = {
        "trends": trends,
        "start_date": start_date.isoformat(),
        "end_date": end_date.isoformat()
    }

    return success_response(
        message="Tendências por categoria geradas com sucesso",
        data=data,
        code=200
    )