from flask_openapi3 import APIBlueprint, Tag
from pydantic import BaseModel
from app.models import Category, db
from app.schemas import CategoryCreate, CategoryUpdate
from app.decorators import token_required
from app.config import Config
from app.errors import api_response, success_response, error_response, ErrorDetail, SuccessResponse, ErrorResponse
from sqlalchemy.exc import IntegrityError

category_tag = Tag(name="Categories", description="Operações com categorias")
category_bp = APIBlueprint("categories", __name__, url_prefix="/api/categories", abp_tags=[category_tag])

@category_bp.post(
    "/",
    responses={
        "201": SuccessResponse,
        "401": ErrorResponse,
        "404": ErrorResponse
    },
    security=[{"bearerAuth": []}]
)
@token_required
@api_response
def create_category(body: CategoryCreate):
    """Cria uma nova categoria"""
    try:
        category = Category(
            name=body.name,
            description=body.description,
            type=body.type,
            user_id=create_category.current_user.id
        )
        
        db.session.add(category)
        db.session.commit()
        
        category_data = {
            "id": category.id,
            "name": category.name,
            "description": category.description,
            "type": category.type,
            "is_fixed": category.is_fixed,
            "created_at": category.created_at_str,
            "updated_at": category.updated_at_str
        }
        
        return success_response(
            message="Categoria criada com sucesso",
            data=category_data,
            code=201
        )
        
    except IntegrityError:
        db.session.rollback()
        return error_response(
            message="Erro ao criar categoria",
            errors=[ErrorDetail(field="name", message="Já existe uma categoria com este nome")],
            code=400
        )

class CategoryQueryParams(BaseModel):
    page: int = 1
    per_page: int = Config.ITEMS_PER_PAGE
    type: str = None

@category_bp.get(
    "/",
    responses={
        "200": SuccessResponse,
        "401": ErrorResponse
    },
    security=[{"bearerAuth": []}]
)
@token_required
@api_response
def list_categories(query: CategoryQueryParams):
    """Lista todas as categorias do usuário (paginado)"""
    query_filter = Category.query.filter_by(user_id=list_categories.current_user.id)
    
    if query.type:
        query_filter = query_filter.filter_by(type=query.type)
    
    pagination = query_filter.paginate(page=query.page, per_page=query.per_page)
    
    items = []
    for category in pagination.items:
        items.append({
            "id": category.id,
            "name": category.name,
            "description": category.description,
            "type": category.type,
            "is_fixed": category.is_fixed,
            "created_at": category.created_at_str,
            "updated_at": category.updated_at_str
        })
    
    data = {
        "items": items,
        "total": pagination.total,
        "page": query.page,
        "per_page": query.per_page,
        "total_pages": pagination.pages
    }
    
    return success_response(
        message="Categorias listadas com sucesso",
        data=data,
        code=200
    )

class CategoryIdPath(BaseModel):
    category_id: int

@category_bp.get(
    "/<int:category_id>",
    responses={
        "200": SuccessResponse,
        "401": ErrorResponse,
        "404": ErrorResponse
    },
    security=[{"bearerAuth": []}]
)
@token_required
@api_response
def get_category(path: CategoryIdPath):
    """Obtém uma categoria específica"""
    category = Category.query.filter_by(
        id=path.category_id,
        user_id=get_category.current_user.id
    ).first_or_404()
    
    category_data = {
        "id": category.id,
        "name": category.name,
        "description": category.description,
        "type": category.type,
        "is_fixed": category.is_fixed,
        "created_at": category.created_at_str,
        "updated_at": category.updated_at_str
    }
    
    return success_response(
        message="Categoria encontrada com sucesso",
        data=category_data,
        code=200
    )

@category_bp.patch(
    "/<int:category_id>",
    responses={
        "200": SuccessResponse,
        "400": ErrorResponse,
        "401": ErrorResponse,
        "404": ErrorResponse
    },
    security=[{"bearerAuth": []}]
)
@token_required
@api_response
def update_category(path: CategoryIdPath, body: CategoryUpdate):
    """Atualiza uma categoria"""
    category = Category.query.filter_by(
        id=path.category_id,
        user_id=update_category.current_user.id
    ).first_or_404()
    
    try:
        if body.name is not None:
            category.name = body.name
        if body.description is not None:
            category.description = body.description
        if body.type is not None:
            category.type = body.type
            
        db.session.commit()
        
        category_data = {
            "id": category.id,
            "name": category.name,
            "description": category.description,
            "type": category.type,
            "is_fixed": category.is_fixed,
            "created_at": category.created_at_str,
            "updated_at": category.updated_at_str
        }
        
        return success_response(
            message="Categoria atualizada com sucesso",
            data=category_data,
            code=200
        )
        
    except IntegrityError:
        db.session.rollback()
        return error_response(
            message="Erro ao atualizar categoria",
            errors=[ErrorDetail(field="name", message="Já existe uma categoria com este nome")],
            code=400
        )

@category_bp.delete(
    "/<int:category_id>",
    responses={
        "200": SuccessResponse,
        "401": ErrorResponse,
        "404": ErrorResponse
    },
    security=[{"bearerAuth": []}]
)
@token_required
@api_response
def delete_category(path: CategoryIdPath):
    """Deleta uma categoria"""
    category = Category.query.filter_by(
        id=path.category_id,
        user_id=delete_category.current_user.id
    ).first_or_404()
    
    try:
        db.session.delete(category)
        db.session.commit()
        
        return success_response(
            message="Categoria deletada com sucesso",
            code=200
        )
        
    except IntegrityError:
        db.session.rollback()
        return error_response(
            message="Não é possível deletar esta categoria",
            errors=[ErrorDetail(field="foreign_key", message="Existem transações vinculadas a esta categoria")],
            code=400
        )