import jwt
from functools import wraps
from datetime import datetime, timedelta
from flask import current_app, request, jsonify
from app.models import User

def generate_token(user_id: int) -> str:
    """Gera um token JWT para o usuário."""
    payload = {
        'user_id': user_id,
        'exp': datetime.utcnow() + current_app.config['JWT_ACCESS_TOKEN_EXPIRES'],
        'iat': datetime.utcnow()
    }
    return jwt.encode(payload, current_app.config['JWT_SECRET_KEY'], algorithm='HS256')

def decode_token(token: str) -> dict:
    """Decodifica um token JWT."""
    try:
        payload = jwt.decode(token, current_app.config['JWT_SECRET_KEY'], algorithms=['HS256'])
        return payload
    except jwt.ExpiredSignatureError:
        raise Exception('Token expirado. Por favor, faça login novamente.')
    except jwt.InvalidTokenError:
        raise Exception('Token inválido. Por favor, faça login novamente.')

def token_required(f):
    """Decorator para proteger rotas que requerem autenticação."""
    @wraps(f)
    def decorated(*args, **kwargs):
        token = None
        auth_header = request.headers.get('Authorization')
        
        if not auth_header:
            return jsonify({'message': 'Token não encontrado'}), 401

        parts = auth_header.split()
        if len(parts) != 2 or parts[0].lower() != 'bearer':
            return jsonify({'message': 'Formato inválido do token. Use: Bearer <token>'}), 401

        token = parts[1]

        try:
            payload = decode_token(token)
            current_user = User.query.get(payload['user_id'])
            
            if not current_user:
                return jsonify({'message': 'Usuário não encontrado'}), 401

            # Adiciona o usuário atual ao contexto da requisição
            setattr(decorated, 'current_user', current_user)  # Para acesso no endpoint
            request.current_user = current_user  # Para acesso via request
            
            return f(*args, **kwargs)
        except Exception as e:
            return jsonify({'message': str(e)}), 401

    return decorated
